// File created: 2007-10-23 13:09:54

package ope.adventure.parsers;

import java.util.InputMismatchException;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ope.adventure.Result;
import ope.adventure.util.Utils;

/* Generic result parser, takes anything of the form:
 * BooleanExpression {
 *		consequence;
 *	}
 *	where there may be any number of consequences
 *	and there must be at least one string as a consequence of its own
 *	(which is the message that is printed when this is the result)
 */
public final class ResultParser {
	private ResultParser() {}

	private static int i;
	public static int getEnd() { return i; }

	public static Result parse(
		final String str,
		final ConsequencePredicate acc
	) {
		final BooleanExpressionParser bexp = new BooleanExpressionParser(str);

		final Result res = new Result();
		res.setCondition(bexp.parse());

		i = Utils.skipWhiteSpace(bexp.getEnd(), str);

		if (str.charAt(i++) != '{')
			throw new BadResultException(
				"'{' expected following boolean expression");

		boolean gotMessage = false;
		while (i < str.length()) {
			i = Utils.skipWhiteSpace(i, str);

			if (str.charAt(i) == '}') {
				++i;
				break;
			}

			if (!gotMessage && str.charAt(i) == '"') {
				int j = ++i;

				while (str.charAt(j) != '"') {
					++j;
					if (j >= str.length())
						throw new BadResultException(
							"hit EOF whilst searching for end of string");
				}

				res.setMessage(str.substring(i, j));

				i = j+1;

				gotMessage = true;
				continue;
			}

			int j = i;
			while (str.charAt(j) != ';') {
				++j;
				if (j >= str.length())
					throw new BadResultException(
						"hit EOF whilst searching for terminating semicolon");
			}

			final String cons = str.substring(i, j);
			if (!acc.accept(cons, res))
				throw new BadResultException("invalid consequence '" +cons+ "'");

			i = j+1;
		}

		if (!gotMessage)
			throw new BadResultException("result lacks message");

		return res;
	}

	public abstract static class ConsequencePredicate {
		public abstract boolean accept(String action, Result r);

		protected final boolean acceptFailure(final String s, final Result res) {
			if (s.equals("fail")) {
				res.setSuccess(false);
				return true;
			} else
				return false;
		}

		// accepts a consequence of the form:
		// str
		protected final boolean acceptJust(
			final String str, final Result.Consequence.Type type,
			final String src, final Result res
		) {
			if (src.equalsIgnoreCase(str)) {
				res.addConsequence(type);
				return true;
			} else
				return false;
		}

		// accepts a consequence of the form:
		// str<whitespace>"<string>"
		protected final boolean acceptStr(
			final String str, final Result.Consequence.Type type,
			final String src, final Result res
		) {
			final String expected = str + "\\p{javaWhitespace}+\"";
			final Matcher m =
				Pattern.compile(expected, Pattern.CASE_INSENSITIVE).matcher(src);

			if (m.lookingAt()) {
				final int start = m.end();

				for (int i = start; i < src.length(); ++i)
				if (src.charAt(i) == '"') {
					res.addConsequence(type, src.substring(start, i));
					return true;
				}
			}
			return false;
		}

		// accepts a consequence of the form:
		// str<whitespace><integer -128 to 127>
		protected final boolean acceptByte(
			final String str, final Result.Consequence.Type type,
			final String src, final Result res
		) {
			final String expected = str + "\\p{javaWhitespace}+";
			final Matcher m =
				Pattern.compile(expected, Pattern.CASE_INSENSITIVE).matcher(src);

			if (m.lookingAt()) {
				res.addConsequence(
					type, new Scanner(
						src.substring(m.end(), src.length())
					).nextByte());

				return true;
			} else
				return false;
		}
	}
}

final class BadResultException extends InputMismatchException {
	public BadResultException(final String msg) {
		super(msg);
	}
}
